unit IDETemplateProject;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs,
  ProjectTemplates, ProjectIntf, LazIDEIntf;

type

  { TTemplateProjectDescriptor }

  TTemplateProjectDescriptor = class(TProjectDescriptor)
  Private
    FTemplates : TProjectTemplates;
    FTemplate : TProjectTemplate;
    FProjectDirectory : String;
    FProjectName : String;
    FVariables : TStrings;
    Function ShowOptionsDialog : TModalResult;
    Function ShowVariableDialog : TModalResult;
  protected
    Procedure InitTemplates; virtual;
    procedure SaveTemplateSettings;
  public
    constructor Create; override;
    destructor destroy; override;
    Function DoInitDescriptor : TModalResult; override;
    function GetLocalizedName: string; override;
    function GetLocalizedDescription: string; override;
    function InitProject(AProject: TLazProject) : TModalResult; override;
    function CreateStartFiles(AProject: TLazProject) : TModalResult; override;
  published
    { Published declarations }
  end;
  
  { TProjectFileDesc }

  TProjectFileDesc = class(TProjectFileDescriptor)
  private
    FIndex : Integer;
    FTemplate : TProjectTemplate;
    FVariables : TStrings;
  public
    constructor Create(ATemplate : TProjectTemplate; Values: TStrings);
    Function GetResourceSource : String;override;
    function CreateSource(const Filename, SourceName,
                          ResourceName: string): string; override;
    function GetLocalizedName: string; override;
    function GetLocalizedDescription: string; override;
  end;


procedure Register;

implementation

uses inifiles,frmCreateFromTemplate,frmtemplateVariables,ConfigStorage;


Var
  TemplateProjectDescriptor : TTemplateProjectDescriptor;

procedure Register;
begin
  TemplateProjectDescriptor:=TTemplateProjectDescriptor.Create;
  RegisterProjectDescriptor(TemplateProjectDescriptor);
end;

{ TTemplateProjectDescriptor }

function TTemplateProjectDescriptor.ShowOptionsDialog : TModalResult;

var
  I: Integer;
  
begin
  with TTemplateOptionsForm.Create(Application) do
    try
      Templates:=Self.FTemplates;
      Result:=ShowModal;
      if Result=mrOK then
        begin
        FProjectDirectory:=IncludeTrailingPathDelimiter(ProjectDir);
        FProjectName:=ProjectName;
        FTemplate:=Template;
        FVariables.Assign(FTemplate.Variables);
        I:=FVariables.IndexOfName('ProjName');
        if (I<>-1) then
          FVariables.Delete(I);
        I:=FVariables.IndexOfName('ProjDir');
        if (I<>-1) then
          FVariables.Delete(I);
        end;
      if SettingsChanged then
        SaveTemplateSettings;
    finally
      Free;
    end;
end;

procedure TTemplateProjectDescriptor.InitTemplates;

Var
  D : String;

begin
  If (FTemplates=Nil) then
    With GetIDEConfigStorage('projtemplate.xml',True) do
      try
        D:=GetValue('TemplateDir',IncludeTrailingPathDelimiter(LazarusIDE.GetPrimaryConfigPath)+'templates');
        FTemplates:=TProjectTemplates.Create(D);
      Finally
        Free;
      end;
end;

procedure TTemplateProjectDescriptor.SaveTemplateSettings;

begin
  With GetIDEConfigStorage('projtemplate.xml',False) do
    try
      SetValue('TemplateDir',FTemplates.TemplateDir);
      WriteToDisk;
    Finally
      Free;
    end;
end;


constructor TTemplateProjectDescriptor.Create;
begin
  inherited Create;
  Name:='Template Project';
  FVariables:=TStringList.Create;
end;

destructor TTemplateProjectDescriptor.destroy;
begin
  FTemplate:=NIl;
  FreeAndNil(FTemplates);
  FreeAndNil(FVariables);
  Inherited;
end;


function TTemplateProjectDescriptor.GetLocalizedName: string;
begin
  Result:='Project from Template';
end;

function TTemplateProjectDescriptor.GetLocalizedDescription: string;
begin
  Result:='Create a project, based on a series of project templates';
end;

function TTemplateProjectDescriptor.ShowVariableDialog : TModalResult;

begin
  With TProjectVariablesForm.Create(Application) do
    try
      Variables:=FVariables;
      Result:=ShowModal;
    Finally
      Free;
    end;
end;

function TTemplateProjectDescriptor.DoInitDescriptor: TModalResult;

begin
  InitTemplates;
  Result:=ShowOptionsDialog;
  If (Result=mrOK) and (FVariables.Count<>0) then
    Result:=ShowVariableDialog;
  If (Result=mrOK) then
    begin
    FVariables.Values['ProjName']:=FProjectName;
    FVariables.Values['ProjDir']:=FProjectDirectory;
    end;
end;


function TTemplateProjectDescriptor.InitProject(AProject: TLazProject) : TModalResult;

Var
  I : Integer;
  AFile: TLazProjectFile;
  FN : String;
  B : Boolean;
  RFN : String;
  L : TStringList;
  
begin
  AProject.AddPackageDependency('FCL');
  AProject.AddPackageDependency('LCL');
  AProject.Title:=FProjectName;
  If Assigned(FTemplate) then
    begin
    FTemplate.CreateProjectDirs(FProjectDirectory,FVariables);
    AProject.ProjectInfoFile:=FProjectDirectory+FProjectName+'.lpi';
    For I:=0 to FTemplate.FileCount-1 do
      begin
      FN:=FTemplate.FileNames[I];
      B:=CompareText(ExtractFileExt(FN),'.lpr')=0;
      If B then
        begin
        FN:=FProjectDirectory+FTemplate.TargetFileName(FN,FVariables);
        AFile:=AProject.CreateProjectFile(FN);
        AFile.IsPartOfProject:=true;
        AProject.AddFile(AFile,Not B);
        AProject.MainFileID:=0;
        L:=TstringList.Create;
        try
          FTemplate.CreateFile(I,L,FVariables);
          AFile.SetSourceText(L.Text);
        Finally
          L.Free;
        end;
        end;
      end;
    Result:=mrOK;
    end
  else
    Result:=mrCancel;
end;

Function TTemplateProjectDescriptor.CreateStartFiles(AProject: TLazProject) : TModalresult;

Const
  IDeOpts =[nfIsPartOfProject,nfOpenInEditor,nfCreateDefaultSrc];

Var
  Descr : TProjectFileDesc;
  I : Integer;
  FN,FN2 : String;
  B : Boolean;
  
begin
  if Assigned(FTemplate) then
    begin
    Result:=mrOK;
    Descr:=TProjectFileDesc.Create(FTemplate,FVariables);
    Try
      For I:=0 to FTemplate.FileCount-1 do
        begin
        FN:=FTemplate.FileNames[I];
        B:=CompareText(ExtractFileExt(FN),'.lpr')<>0;
        If B then
          begin
          B:=CompareText(ExtractFileExt(FN),'.lfm')<>0;
          If B then
            begin
            FN2:=ChangeFileExt(FN,'.lfm');
            B:=FileExists(FN2);
            FN:=FProjectDirectory+FTemplate.TargetFileName(FN,FVariables);
            If B then
              Descr.ResourceClass:=TForm;
            Descr.FIndex:=I;
            LazarusIDE.DoNewEditorFile(Descr,FN,'',IdeOpts)
            end;
          end;
        end;
    Finally
      Descr.Free;
    end;
    end
  else
    Result:=mrCancel;
end;

{ TProjectFileDesc }


constructor TProjectFileDesc.Create(ATemplate: TProjectTemplate; Values :TStrings);
begin
  Inherited Create;
  FTemplate:=ATemplate;
  FVariables:=Values;
  Name:='Regular File';
end;

function TProjectFileDesc.CreateSource(const Filename, SourceName,
  ResourceName: string): string;

Var
  L : Tstrings;
  
begin
  L:=TstringList.Create;
  try
    FTemplate.CreateFile(FIndex,L,FVariables);
    Result:=L.Text;
  Finally
    L.Free;
  end;
end;

function TProjectFileDesc.GetResourceSource: String;

Var
  L : Tstrings;
  FN : String;

begin
  Result:='';
  If (ResourceClass<>Nil) then
    begin
    L:=TstringList.Create;
    try
      FN:=ChangeFileExt(FTemplate.FileNames[FIndex],'.lfm');
      FTemplate.CreateFile(FN,L,FVariables);
      Result:=L.Text;
    Finally
      L.Free;
    end;
    end;
end;

function TProjectFileDesc.GetLocalizedName: string;
begin
  Result:='Template unit source';
end;

function TProjectFileDesc.GetLocalizedDescription: string;
begin
  Result:='Template unit source';
end;


end.
