#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <cbm.h>
#include <peekpoke.h>
#include <string.h>
#include <6502.h>

#define SET_ST(x) { POKE (0x90,x); }
#define GET_ST() (PEEK(0x90))

#define TAPE_BUFFER ((char *) 0x033c)
#define FLASH_BUFFER ((char *) 0xc000)

#define flash_data           0xde00
#define flash_data_deselect  0xde01
#define flash_data_dummy     0xde02

#define BOARD_BELUGA         1
#define BOARD_SUPERGUPPY     2

const char cbm80[] = {0x30, 0x38, 0xcd, 0xc2, 0xc3};

char flash_mfg = 0xff;
char flash_dev = 0xff;
char flash_cap = 0xff;
char device = 0xff;
char board_type;

char erase_full;

void strout(register const char *s) {
  while (*s!=0) {
    cbm_k_bsout(*s);
    ++s;
  }
}

void numout(unsigned v) {
  utoa(v,TAPE_BUFFER,10);
  strout(TAPE_BUFFER);
}

void input_str() {
  register char *s;
  s=TAPE_BUFFER;
  do {
    *s=cbm_k_basin();
  } while (*s++!=13);
  *s=0;
}

char input_yn() {
  register char *s = TAPE_BUFFER;

  do {
    strout("(y/n) ");
    input_str();

    while (*s==' ')
      s++;
    cbm_k_bsout('\n');
    if (*s=='y')
      return 1;
    if (*s=='n')
      return 0;
  } while (1);
}

char input_num() {
  register char *s = TAPE_BUFFER;

  do {
    input_str();

    while (*s==' ')
      s++;
    cbm_k_bsout('\n');
    if (*s>='0' && *s<='9')
      return *s - '0';
  } while (1);
}

void handle_st(const char st) {
  if (st & 128)
    strout("Device not present");
  if (st & 64)
    strout("End of file");
  if (st & 16)
    strout("Verify");
  if (st & 2)
    strout("Read");
  if (st & 1)
    strout("Write ");
  if (st)
    strout(" error.\n");
}

void serialout(register const char *s) {
  while (*s!=0) {
    cbm_k_ciout(*s);
    ++s;
  }
}

void file_open(const char *s, const char chan) {
  cbm_k_listen(device);
  cbm_k_second(0xf0 | chan);
  serialout(s);
  cbm_k_unlsn();
}

void file_close(const char chan) {
 /* Tell device we want to close the directory. */
  cbm_k_listen(device);
  cbm_k_second(0xe0 | chan);
  cbm_k_unlsn();
}

char flash_read_byte_spi() {
  static char b,d;

  d=PEEK(flash_data);
  b = (d & 0x20 ? 0x80 : 0x00) | (d & 0x02 ? 0x40 : 0x00);
  d=PEEK(flash_data);
  b |= (d & 0x20 ? 0x20 : 0x00) | (d & 0x02 ? 0x10 : 0x00);
  d=PEEK(flash_data);
  b |= (d & 0x20 ? 0x08 : 0x00) | (d & 0x02 ? 0x04 : 0x00);
  d=PEEK(flash_data);
  b |= (d & 0x20 ? 0x02 : 0x00) | (d & 0x02 ? 0x01 : 0x00);
  return b;
}

char flash_read_endbyte_spi() {
  static char b,d;

  d=PEEK(flash_data);
  b = (d & 0x20 ? 0x80 : 0x00) | (d & 0x02 ? 0x40 : 0x00);
  d=PEEK(flash_data);
  b |= (d & 0x20 ? 0x20 : 0x00) | (d & 0x02 ? 0x10 : 0x00);
  d=PEEK(flash_data);
  b |= (d & 0x20 ? 0x08 : 0x00) | (d & 0x02 ? 0x04 : 0x00);
  d=PEEK(flash_data_deselect);
  b |= (d & 0x20 ? 0x02 : 0x00) | (d & 0x02 ? 0x01 : 0x00);
  return b;
}

void flash_write_byte_spi(char b) {
  POKE(flash_data,0xee | ((b & 0x80) ? 0x10 : 0x00) | ((b & 0x40) ? 0x01 : 0x00));
  POKE(flash_data,0xee | ((b & 0x20) ? 0x10 : 0x00) | ((b & 0x10) ? 0x01 : 0x00));
  POKE(flash_data,0xee | ((b & 0x08) ? 0x10 : 0x00) | ((b & 0x04) ? 0x01 : 0x00));
  POKE(flash_data,0xee | ((b & 0x02) ? 0x10 : 0x00) | ((b & 0x01) ? 0x01 : 0x00));
}

void flash_write_endbyte_spi(char b) {
  POKE(flash_data,0xee | ((b & 0x80) ? 0x10 : 0x00) | ((b & 0x40) ? 0x01 : 0x00));
  POKE(flash_data,0xee | ((b & 0x20) ? 0x10 : 0x00) | ((b & 0x10) ? 0x01 : 0x00));
  POKE(flash_data,0xee | ((b & 0x08) ? 0x10 : 0x00) | ((b & 0x04) ? 0x01 : 0x00));
  POKE(flash_data_deselect,0xee | ((b & 0x02) ? 0x10 : 0x00) | ((b & 0x01) ? 0x01 : 0x00));
}

void flash_wren() {
  flash_write_byte_spi(0x06);
}

void flash_wait_wip() {
  static char d;
  /* Wait while write in progress */
  flash_write_byte_spi(0x05);
  do {
    d=PEEK(flash_data);
    d=PEEK(flash_data);
    d=PEEK(flash_data);
    d=PEEK(flash_data);
  } while (d & 0x02);
  d=PEEK(flash_data_deselect);
}

void flash_chip_erase() {
  /* WREN */
  flash_write_endbyte_spi(0x06);
  /* Chip erase */
  flash_write_endbyte_spi(0x60);

  flash_wait_wip();
}

void flash_sector_erase(char sa_h, char sa_m) {
  /* WREN */
  flash_write_endbyte_spi(0x06);
  /* Erase sector */
  flash_write_byte_spi(0x20);
  flash_write_byte_spi(sa_h);
  flash_write_byte_spi(sa_m);
  flash_write_endbyte_spi(0x00);
  flash_wait_wip();
};

void flash_sector_program(char sa_h, char sa_m, char* data) {
  static char i,j;

  for (i=16;i!=0;--i) {
    /* WREN */
    flash_write_endbyte_spi(0x06);
    /* Program page sector */
    flash_write_byte_spi(0x32);
    flash_write_byte_spi(sa_h);
    flash_write_byte_spi(sa_m);
    flash_write_byte_spi(0x00);
    for (j=0;j<255;++j)
      POKE(flash_data,data[j]);
    POKE(flash_data_deselect,data[255]);
    ++sa_m;
    if (sa_m==0)
      ++sa_h;
    data+=0x100;
    flash_wait_wip();
  }
}

char flash_read_status2() {
  flash_write_byte_spi(0x35);
  return flash_read_endbyte_spi();
}

void flash_write_status2(char s) {
  /* WREN */
  flash_write_endbyte_spi(0x06);
  /* Write status register 2 */
  flash_write_byte_spi(0x31);
  flash_write_endbyte_spi(s);
}


char check_boot_sector() {
  static char b,i;

  flash_write_byte_spi(0xeb);
  /* Address */
  POKE(flash_data,0x00);
  POKE(flash_data,0x00);
  POKE(flash_data,0x00);
  /* M byte */
  POKE(flash_data,0x00);
  /* Dummy */
  asm("nop");
  POKE(0xff,PEEK(flash_data));
  asm("nop");
  POKE(0xff,PEEK(flash_data));

  b = PEEK(flash_data);
  if (b==0xff) goto nok;
  for (i=0;i<7;i++)
    if (PEEK(flash_data)!=b) {
      strout("cfg\n");
      goto nok;
    }

  for (i=0;i<5;i++)
    if (PEEK(flash_data)!=cbm80[i]) {
      strout("cbm80\n");
      goto nok;
    }

   b = PEEK(flash_data_deselect);
   return 1;
nok:
 b = PEEK(flash_data_deselect);
 return 0;
}

void install_initsram() {
  register char i,j,sa;

  SET_ST(0);
  strout("Programming initial SRAM:\n");
  file_open("initsram,s,r",2);
  if (GET_ST())
    goto file_error;

  cbm_k_talk(device);
  cbm_k_tksa(0x62);
  if (GET_ST())
    goto file_error;

  for (sa=16;sa!=80;sa+=16) {
    if (!erase_full) {
      strout("Erasing sector at ");
      numout(sa);
      cbm_k_bsout('\n');
      flash_sector_erase(0,sa);
    }
    SET_ST(0);
    for (i=0;i!=16;i++) {
      j=0;
      do {
        if (GET_ST())
          goto file_error;
        FLASH_BUFFER[(i<<8) + j]=cbm_k_acptr();
        j++;
      } while (j!=0);
    }

    strout("Programming sector at ");
    numout(sa);
    cbm_k_bsout('\n');
    flash_sector_erase(0,sa);
    flash_sector_program(0,sa,FLASH_BUFFER);
  }

  goto end;

file_error:
//  file_close(2);
  handle_st(GET_ST());
  goto end;
flash_error:
  strout("Flash write error\n");
end:
  cbm_k_untlk();
//  fast1541_enable_screen();
//  filename_extra[0]=0;
  file_close(2);
}

void install_boot_sector() {
  register char i,j;

  SET_ST(0);
  if (board_type==BOARD_BELUGA)
    file_open("belugaboot,s,r",2);
  else
    file_open("sguppyboot,s,r",2);
  if (GET_ST())
    goto file_error;

  cbm_k_talk(device);
  cbm_k_tksa(0x62);
  if (GET_ST())
    goto file_error;

  if (!erase_full) {
    strout("Erasing bootsector...\n");
    flash_sector_erase(0,0);
  }
  SET_ST(0);
  for (i=0;i!=16;i++) {
    j=0;
    do {
      if (GET_ST())
        goto file_error;
      FLASH_BUFFER[(i<<8) + j]=cbm_k_acptr();
      j++;
    } while (j!=0);
  }

  strout("Programming boot sector\n");
  flash_sector_program(0,0,FLASH_BUFFER);

  goto end;

file_error:
//  file_close(2);
  handle_st(GET_ST());
  goto end;
flash_error:
  strout("Flash write error\n");
end:
  cbm_k_untlk();
//  fast1541_enable_screen();
//  filename_extra[0]=0;
  file_close(2);
}

void read_jedec_id() {
  flash_write_byte_spi(0x9f);
  flash_mfg=flash_read_byte_spi();
  flash_dev=flash_read_byte_spi();
  flash_cap=flash_read_endbyte_spi();
}

void main () {
  static char s;

  device = PEEK(0xba);

  /*  Ask board type */
  strout("Beluga / Super Guppy installer\n\n");
  
  do {
    strout("Which cartridge do you want to install?\n");
    strout("1 - Beluga\n");
    strout("2 - Super Guppy\n");
    strout("Your choice: ");
    board_type=input_num();
  } while (board_type==0 || board_type>2);

  /* Make sure the flash is in SPI mode. */
  POKE(flash_data_deselect,0xff);
  POKE(flash_data_deselect,0xff);

  strout("Detecting flash... \n");
  read_jedec_id();
  strout("\nManufacturer: ");
  numout(flash_mfg);
  strout("\nDevice: ");
  numout(flash_dev);
  strout("\nCapacity: 1^");
  numout(flash_cap);
  strout("\nChecking quad enable: ");
  s = flash_read_status2();
  if ((s & 0x02) == 0) {
    strout("off\n");
    strout("Shall I Switch quad enable on? ");
    if (input_yn()==0) {
      strout("Cannot continue.\n");
      exit(1);
    };
    flash_write_status2(s | 0x02);
    s = flash_read_status2();
    if (s & 0x02 == 0) {
      strout("Failed switching quad enable on!! Aborting.\n");
      exit(1);
    }
    strout("Done\n");
  } else {
    strout("on\n");
  }
  strout("Would you like to erase the full chip? ");
  erase_full = input_yn();
  if (erase_full) {
    flash_chip_erase();
    strout("Done\n");
  }
  strout("Boot sector looks ");
  if (check_boot_sector())
    strout("valid.\n");
  else
    strout("invalid\n");
  strout("Would you like to install the boot sector? ");
  if (input_yn())
    install_boot_sector();
  strout("Would you like to install the initial SRAM content? ");
  if (input_yn())
    install_initsram();
  cbm_k_bsout('\n');
}
