
.macro stackpush value
		lda value
		pha
		nop ; Phantom
.endmacro

.segment "stage1_2"

.include "stage2_sz.inc"

;
; This is the value of the Beluga configuration register that is programmed
; during reset by the Beluga controller.
;

.ifdef BELUGA
startup_config = $85
.endif
.ifdef SUPERGUPPY
startup_config = $80
.endif


bootconfig:
.repeat 8
    .byte startup_config
.endrep

;
; The C64 checks the CBM80 signature in reverse. Because the Beluga
; uses sequential memory (the address is ignored), this means we need to
; present the signature in reverse.
;
CBM80:
    .byte $30,$38,$CD,$C2,$C3

;
; This is the address the C64 jumps to after discovering the CBM80 signature.
;

coldstart:
    .word $8000

; This is the stage 1 boot. At this point we can only execute speedcode
; (because we have no address bus, jumps are impossible).
; Exuting SAM speecode is slightly different from normal speedcode,
; because some 6502 instruction perform phantom read cycles, i.e. the
; CPU reads another byte that is discarded. In such case we will send a 
; NOP (no NOP is executed by the CPU).


boot_stage_1:
	lda #$01
	sta $d020
	ldx #stage2_size - 1
	txs
	nop ; Phantom
.include "stage1_stackpush.inc"
    inc $d020
    jmp $0100


.segment "stage3"
:   lda $de00
    sta $0500,x
    inx
    bne :-

	ldx $de01
	; Now tell the flash to enter QPI mode by sending command $38
	; The flash is in SPI mode and accepts two bits at a time
	; $38 = %00 11 10 00... send $00 $ff $f0 $00
	lda #$00
	sta $de00
	ldx #$ff
	stx $de00
	ldx #$f0
	stx $de00
	sta $de01 ; Write and deselect flash

	; The flash is now in QPI mode, so we now can send commands to flash
	; memory without encoding them two bits at a time, we now can send them
	; the way they are.

	; We now want to configure 4 dummy cycles with the $C0 (set read
	; parameters) opcode. Unfortunately the evil manufacturers have
	; managed to be incompatible which value corresponds to 4 dummy
	; cycles. For example Winbod requires $10 while Gigadevice requires $00.
	; We could detect the manufacturer and do an "if manufacturer"
	; but it is better to autodetect which value works.
	;
	; In order to do so, we will try to read at flash location $000007.
	; If the amount of dummy cycles is correct, we will read $85. If it
	; is too many, we will read $ff, if it is too low we read a byte
	; from the CBM80 signature, which is <> $85.
	;
	lda #$00
:	ldx #$c0		; Set read parameters
	stx $de00
	sta $de01
	ldx #$eb		; Fast read quad IO
	stx $de00
	ldx #$00
	stx $de00
	stx $de00
	ldx #$07
	stx $de00
	bit $de00       ; Do not use $de02 while dummy cycles not correct
	bit $de00       ; It could cause bus conflict with VIC-II
	ldx $de01
	cpx #startup_config		; Read was correct?
	beq :+
	clc
	adc #$10		; Try next possible value
	cmp #$40
	bne :-
	jmp fail

:
.ifdef BELUGA
	sta $df00		; Keep correct value in SRAM
	inc $d020
	;
	; We now will initialize the SRAM. We will read 16KB from flash address
	; $001000 and store it into the SRAM
	;
	lda #$83        ; Make writeable SRAM visible at $8000 and $e000
	sta $de03
	
	lda #$eb
	sta $de00
	ldx #$00
	stx $de00
	lda #$10
	sta $de00
	stx $de00
	bit $de02
	
@2:	ldx #$00
@1:	lda $de00
@a:	sta $8000,x
	inx
	bne @1
	
	ldy @a+2
	iny
	beq @3
	sty @a+2
@b:	cpy #$a0
	bcc @2
	ldy #$e0
	sty @a+2
	ldy #$18 ; Opcode for clc
	sty @b
	sty @b+1
	bcs @2
@3:
    ldx $de01 ; deselect flash

    inc $d020
    ;
    ; We will run OpenROM kernal with CBM BASIC, but due to license incompatibility
    ; BASIC routines at $E000 cannot be redistributed, so we copy them at runtime
    ; from the real kernal into SRAM.
    ;

@6:	lda #$80
	sta $de03
	ldx #$00
@4:	lda $e000,x
	sta $0600,x
	inx
@7:	cpx #$00
	bne @4
	ldx #$00  ; Not superfluous!
	lda #$83
	sta $de03
@9:	lda $0600,x
@5:	sta $e000,x
	inx
@8:	cpx #$00
	bne @9

	ldy @4+2
	iny
	cpy #$e5
	beq @z
	cpy #$e4
	sty @4+2
	sty @5+2
	bne @6
	lda #$73
	sta @7+1
	sta @8+1
	bne @6

@z:	; JiffyDOS has some patches in BASIC to call into modifies ROM. Because we are
    ; copying from the ROM in the machine and our Open KERNAL does not have the
    ; JiffyDOS extensions, this will fail. Therefore we unpatch JiffyDOS from BASIC.
    lda #$ba
    sta $e1de
    lda #$ff
    sta $e1df
    lda #$53
    sta $e395
    lda #$8b
    sta $e447
    lda #$e3
    sta $e448
    lda #$7c
    sta $e44b
    lda #$a5
    sta $e44c
    ;

    inc $d020
	; We now switch the C64 to normal mode (to disable SAM from memory map).
	; This also turns off cartridge led.
	ldy #$00
	; As we have no nice software on the cartridge yet, let's just continue
	; normal startup of the C64.
    ; Prepare the CIA to scan the keyboard
    lda #$7f
    sta $dc00   ; pull down row 7 (DPA)
    ldx #$ff
    stx $dc02   ; DDRA $ff = output (X is still $ff from copy loop)
    inx
    stx $dc03   ; DDRB $00 = input

    ; Read the keys pressed on this row
    lda $dc01   ; read coloumns (DPB)
    ; (leave the CIA registers as they are, IOINIT will initialize them properly)

    ; Check if one of the magic kill keys was pressed
    and #$e0    ; only leave "Run/Stop", "Q" and "C="
    cmp #$e0
    bne @kill    ; branch if one of these keys is pressed
    ldy #$03
@kill:
    sty $de03
	jmp $fce2
.endif

.ifdef SUPERGUPPY
    inc $d020
    ; We now switch the C64 to normal mode (to disable SAM from memory map).
    ; This also turns off the cartridge led
    lda #$30
    sta $de04
	; As we have no nice software on the cartridge yet, let's just continue
	; normal startup of the C64.
	jmp $fce2
.endif

fail:
    lda #15
    sta $d020
    ; Blink the led
.ifdef BELUGA
    lda $de03
    eor #$80
    sta $de03
.endif
.ifdef SUPERGUPPY
    lda $de04
    eor #$80
    sta $de04
.endif
    ldy #40
    ldx #0
:   dex
    bne :-
    dey
    bne :-
    jmp fail
